/*
		Display program for BL files, which record broadcast server activity.
		This code is not particularly portable. It was written for Microsoft C under DOS.
		It will also compile under Quick C for windows, and the unixtime module sorts
		out the problem that ANSI standard time is not the same as good old unix time.

		fFixed - 	This flag is FALSE for early UoSAT-3 and UoSAT-5 BL files because there was
						  a bug in the onboard logging task wich caused incorrect counts in the
						  entries for overwritten requests and un-fresh requests.

		Original: Jeff Ward, Surrey Satellite Technology, Ltd.
*/
/*
		History:
		19 April 1993
		Added ability to read the version from the log file if the first entry
		in the log file has length 2. Otherwise version is assumed to be
		version 1.
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "unixtime.h"
#include "blogdisp.h"

#define MAGIC 25569.0

struct STAT_STRUCT stats;
struct STAT_STRUCT statDaily;
int fBytemode = FALSE;													/* Display byte counts */
int fCmdmode = FALSE;														/* Display command counts */
int fDirmode = FALSE;														/* Display dir counts */
int fFixed = TRUE;                              /* Problem with some byte counts */

void main(int argc, char * argv[]){
	FILE * pFile;
	time_t tim;
	int len;
	double timSerial;															/* Microsoft time serial no.			 	*/
	int fNewCSV = FALSE;													/* True when building new CSV file 	*/
	int fInited = FALSE;													/* Set after reading first record		*/
	int version;

	/* Struct to store daily totals */
	memset(&statDaily, 0, sizeof(struct STAT_STRUCT));

	/* Insufficient args. Show help */
	if (argc<2){
		fprintf(stderr,"blogdisp <blname> [<-b | -c | -d> [<csvname>]]  Broadcast log display program.\n");
		fprintf(stderr, "Version 1.6\n");
		fprintf(stderr,"<blname> is the name of the BL log input file.\n");
		fprintf(stderr,"-b to display byte counters.\n");
		fprintf(stderr,"-c to display command counters.\n");
		fprintf(stderr,"-d to display directory counters.\n");
		fprintf(stderr,"<csvname> is name of a comma separated value output file for spreadsheets.\n");
		exit(1);
	}
	/* Check second arg for display flag */
	if (argc > 2){
		strlwr(argv[2]);
		if (!strcmp(argv[2], "-b"))
			fBytemode = TRUE;
		else if (!strcmp(argv[2], "-c"))
			fCmdmode = TRUE;
		else if (!strcmp(argv[2], "-d"))
			fDirmode = TRUE;
	}
	/* Or default to command counter display */
	else
		fCmdmode = TRUE;

  /* First arg supplies input file name */
	pFile = fopen(argv[1], "rb");
	if (pFile){
		version = GetVer(pFile);
		if (version > 0){

			while (! feof(pFile)){
				if (fread(&len, sizeof(int), 1, pFile)) {
					if (fread(&tim, sizeof(tim), 1, pFile)) {
						if (!fInited){
							fInited = TRUE;
							fFixed = IsNewLog(tim);
						}
						if (fread(&stats, len-sizeof(tim), 1, pFile)){
							process(&tim, &stats, version);
							sum(&stats, &statDaily, version);
						}
					}
				}
			}
		}
		else{
			printf("Error reading version ID from log file.\n");
		}
		fclose(pFile);

		/* 	Possible third arg supplies output file name for comma	*/
		/*	separated value output for spreadsheets.								*/
		if (argc == 4){

			/* Determine whether this is a new .CSV file or one to		*/
			/* append to.																							*/
			pFile = fopen(argv[3], "rt");
			if (pFile == NULL)
				fNewCSV = TRUE;
			else{
				fNewCSV = FALSE;
				fclose(pFile);
			}

			pFile = fopen(argv[3], "at");
			if (pFile){
				/* Put top in .CSV file if new */
				if (fNewCSV){
					fprintf(pFile, "date,Start Cmds,Hole Fills,End Cmds,"
												 "No -2,No -1,No -3,"
												 "Bytes requested,Bytes transmitted,Ticks,Overwritten,"
												 "Too old,Ended,Fopen Error,PFH Error,Dir Requests,"
												 "Bytes of Dirs Txd,Ticks,Stations Heard,Eof,Long cmds\n");
				}
				/* Remove bogus values if not in a fixed log */
				if (!fFixed){
					statDaily.nbOverwrite = 0l;
					statDaily.nbUnfresh = 0l;
				}
				timSerial = ((double) tim / (24.0*60.0*60.0)) + MAGIC;
				fprintf(pFile, "%8.4f,%ld,%ld,%ld,"
											 "%ld,%ld,%ld,%ld,"
											 "%ld,%ld,%ld,%ld,"
											 "%ld,%ld,%ld,%ld,%ld,%ld,%d,%ld,%d\n",
					timSerial,
					statDaily.nStartFile,
					statDaily.nHolefills,
					statDaily.nEndFile,
					statDaily.nNoFile,
					statDaily.nNoRoom,
					statDaily.nNotOK,
				  statDaily.nbRequested,
					statDaily.nbTransmitted,
					statDaily.TicksData,
					statDaily.nbOverwrite,
					statDaily.nbUnfresh,
					statDaily.nbEnd,
					statDaily.nbFopenErr,
					statDaily.nbPfhErr,
					statDaily.nDirReqs,
					statDaily.nbDirTxd,
					statDaily.TicksDir,
					statDaily.nNewStns,
					statDaily.nbEof,
					statDaily.nLongFile
					);
				fclose(pFile);
			}
		}
	}
	/* No such file as first argument specifies */
	else{
		fprintf(stderr, "blogdisp - Error: input file does not exist.\n");
		exit(2);
	}
}

int GetVer(FILE * pFile){
	int ver=-1;
	int len;
	if (fread(&len, sizeof(int), 1, pFile)) {
		if (len == 2) {
			fread(&ver, 2, 1, pFile);
		}
		else {
			ver = 1;
			fseek(pFile, 0L, SEEK_SET);
		}
	}
	return ver;
}
/*-----------------------------------------------*/
/* Add present stats values to the running total */
/*-----------------------------------------------*/
void sum(struct STAT_STRUCT * pStats, struct STAT_STRUCT * pTotal, int version){
	pTotal->nStartFile += pStats->nStartFile;
	pTotal->nHolefills  += pStats->nHolefills;
	pTotal->nEndFile += pStats->nEndFile;
	pTotal->nNoFile += pStats->nNoFile;
	pTotal->nNoRoom += pStats->nNoRoom;
	pTotal->nNotOK += pStats->nNotOK;
  pTotal->nbRequested  += pStats->nbRequested;
	pTotal->nbTransmitted  += pStats->nbTransmitted;
	pTotal->nbOverwrite  += pStats->nbOverwrite;
	pTotal->nbUnfresh  += pStats->nbUnfresh;
	pTotal->nbEnd  += pStats->nbEnd;
	pTotal->nbFopenErr  += pStats->nbFopenErr;
	pTotal->nbPfhErr  += pStats->nbPfhErr;
	pTotal->nDirReqs += pStats->nDirReqs;
	pTotal->nbDirTxd += pStats->nbDirTxd;
	if (version > 1){
		pTotal->TicksDir += pStats->TicksDir;
		pTotal->TicksData += pStats->TicksData;
		pTotal->nNewStns += pStats->nNewStns;
		if (version>2){
			pTotal->nbEof += pStats->nbEof;
			pTotal->nLongFile += pStats->nLongFile;
		}
	}
}

/*------------------------------------------------*/
/* Display whichever values the user wanted to 		*/
/* display, as indicated by display flags.				*/
/*------------------------------------------------*/
int process(time_t * ptim, struct STAT_STRUCT * pStats, int version){
	static int first=1;
	struct tm * ptm;
	long residue;
	ptm = gmtime(ptim);
	if (first){
		fprintf(stdout, "Broadcast activity %02d/%02d/%02d - ", ptm->tm_mday, ptm->tm_mon+1, ptm->tm_year);

		if (fCmdmode)
			fprintf(stdout, "(Commands)\n");
		else if (fBytemode)
			fprintf(stdout, "(Bytes)\n");
		else if (fDirmode)
			fprintf(stdout, "(Directories)\n");

		fprintf(stdout, "Log is version %d.\n",version);

		if (fBytemode){
			if (fFixed)
				fprintf(stdout, "HH:MM   Req'd     Tx'd    OvrWr     TooOld    Ended    Fopen   PFHerr   EOF\n");
			else
				fprintf(stdout, "HH:MM   Req'd     Tx'd    Ended    Fopen   PFHerr\n");
		}
		else if (fCmdmode){
			fprintf(stdout, "HH:MM Start  Fill   End  Long  NO-1  NO-2  NO-3  Stations\n");
		}
		else if (fDirmode){
			fprintf(stdout, "HH:MM Requests  Bytes Tx'd\n");
		}
		first = 0;
	}
	fprintf(stdout, "%02d:%02d ", ptm->tm_hour, ptm->tm_min);
	if (fCmdmode){
		fprintf(stdout, "% 5ld ", pStats->nStartFile);
		fprintf(stdout, "% 5ld ", pStats->nHolefills);
		fprintf(stdout, "% 5ld ", pStats->nEndFile);
		fprintf(stdout, "% 5d ", pStats->nLongFile);
		fprintf(stdout, "% 5ld ", pStats->nNoFile);
		fprintf(stdout, "% 5ld ", pStats->nNoRoom);
		fprintf(stdout, "% 5ld ", pStats->nNotOK);
		fprintf(stdout, "% 5d", pStats->nNewStns);
	}
	else if (fBytemode){
		fprintf(stdout, "%8ld ", pStats->nbRequested);
		fprintf(stdout, "%8ld ", pStats->nbTransmitted);
		if (fFixed) {
			fprintf(stdout, "%10lu ", pStats->nbOverwrite);
			fprintf(stdout, "%8ld ", pStats->nbUnfresh);
		}
		fprintf(stdout, "%8ld ", pStats->nbEnd);
		fprintf(stdout, "%8ld ", pStats->nbFopenErr);
		fprintf(stdout, "%8ld ", pStats->nbPfhErr);
		fprintf(stdout, "%8ld ", pStats->nbEof);
		residue = pStats->nbRequested - (pStats->nbTransmitted +
							pStats->nbOverwrite + pStats->nbUnfresh +
							pStats->nbEnd + pStats->nbFopenErr + pStats->nbPfhErr +
							pStats->nbEof);
		if (fFixed)
			fprintf(stdout, " (%7ld)", residue);

	}
	else if (fDirmode){
		fprintf(stdout, "% 5ld  ", pStats->nDirReqs);
		fprintf(stdout, "% 8ld ", pStats->nbDirTxd);
	}
	fprintf(stdout, "\n");

  return 1;
}

/*----------------------------------------------------------------------*/
/* Checks to see if this was one of the early logs which will have bad	*/
/* total byte counts in it.																							*/
/* UO-22 files from 6 March 1992 are OK.																*/
/* UoSAT-3 files have yet to be fixed 15 April 1992.										*/
/*----------------------------------------------------------------------*/
int IsNewLog(time_t tim){
	time_t TimFix;
	struct tm Tm;
	memset(&Tm, 0, sizeof(Tm));
	Tm.tm_year = 92;
	Tm.tm_mon = 2;
	Tm.tm_mday = 6;
	TimFix = mktime(&Tm);
	if (tim > TimFix)
		return(TRUE);
	else
		return(FALSE);
}
