/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.cassandra.index.sai.cql;

import org.junit.BeforeClass;
import org.junit.Test;

import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.index.sai.SAITester;
import org.apache.cassandra.service.StorageService;

public class EmptyStringLifecycleTest extends SAITester
{
    @BeforeClass
    public static void setup()
    {
        StorageService.instance.initServer(); // Ensure the node has advanced out of STARTING mode
    }

    @Test
    public void testBeforeAndAfterFlush()
    {
        createTable("CREATE TABLE %s (k int PRIMARY KEY, v text)");
        disableCompaction(KEYSPACE);
        createIndex(String.format(CREATE_INDEX_TEMPLATE, 'v'));

        execute("INSERT INTO %s (k, v) VALUES (0, '')");
        execute("INSERT INTO %s (k) VALUES (1)");

        UntypedResultSet rows = execute("SELECT * FROM %s WHERE v = ''");
        assertRows(rows, row(0, ""));

        flush();
        rows = execute("SELECT * FROM %s WHERE v = ''");
        assertRows(rows, row(0, ""));
    }

    @Test
    public void testAfterInitialBuild()
    {
        createTable("CREATE TABLE %s (k int PRIMARY KEY, v text)");
        disableCompaction(KEYSPACE);

        execute("INSERT INTO %s (k, v) VALUES (0, '')");
        execute("INSERT INTO %s (k) VALUES (1)");
        createIndex(String.format(CREATE_INDEX_TEMPLATE, 'v'));

        UntypedResultSet rows = execute("SELECT * FROM %s WHERE v = ''");
        assertRows(rows, row(0, ""));
    }

    @Test
    public void testAfterCompaction()
    {
        createTable("CREATE TABLE %s (k int PRIMARY KEY, v text)");
        disableCompaction(KEYSPACE);
        createIndex(String.format(CREATE_INDEX_TEMPLATE, 'v'));

        execute("INSERT INTO %s (k, v) VALUES (0, '')");
        execute("INSERT INTO %s (k) VALUES (1)");
        flush();

        execute("INSERT INTO %s (k, v) VALUES (1, '')");
        flush();
        
        compact();

        UntypedResultSet rows = execute("SELECT * FROM %s WHERE v = ''");
        assertRows(rows, row(1, ""), row(0, ""));
    }
}
