/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.spark.sql.comet

import org.apache.spark.rdd.RDD
import org.apache.spark.sql.catalyst.expressions.Attribute
import org.apache.spark.sql.catalyst.plans.physical.{Partitioning, SinglePartition, UnknownPartitioning}
import org.apache.spark.sql.execution.{CoalesceExec, SparkPlan, UnaryExecNode}
import org.apache.spark.sql.vectorized.ColumnarBatch

import com.google.common.base.Objects

import org.apache.comet.{CometConf, ConfigEntry}
import org.apache.comet.serde.OperatorOuterClass
import org.apache.comet.serde.operator.CometSink

object CometCoalesceExec extends CometSink[CoalesceExec] {

  override def enabledConfig: Option[ConfigEntry[Boolean]] = Some(
    CometConf.COMET_EXEC_COALESCE_ENABLED)

  override def createExec(
      nativeOp: OperatorOuterClass.Operator,
      op: CoalesceExec): CometNativeExec = {
    CometSinkPlaceHolder(
      nativeOp,
      op,
      CometCoalesceExec(op, op.output, op.numPartitions, op.child))
  }
}

/**
 * This is basically a copy of Spark's CoalesceExec, but supports columnar processing to make it
 * more efficient when including it in a Comet query plan.
 */
case class CometCoalesceExec(
    override val originalPlan: SparkPlan,
    override val output: Seq[Attribute],
    numPartitions: Int,
    child: SparkPlan)
    extends CometExec
    with UnaryExecNode {
  protected override def doExecuteColumnar(): RDD[ColumnarBatch] = {
    val rdd = child.executeColumnar()
    if (numPartitions == 1 && rdd.getNumPartitions < 1) {
      // Make sure we don't output an RDD with 0 partitions, when claiming that we have a
      // `SinglePartition`.
      CometExecUtils.emptyRDDWithPartitions(sparkContext, 1)
    } else {
      rdd.coalesce(numPartitions, shuffle = false)
    }
  }

  override def outputPartitioning: Partitioning = {
    if (numPartitions == 1) SinglePartition
    else UnknownPartitioning(numPartitions)
  }

  override protected def withNewChildInternal(newChild: SparkPlan): SparkPlan =
    this.copy(child = newChild)

  override def equals(obj: Any): Boolean = {
    obj match {
      case other: CometCoalesceExec =>
        this.numPartitions == other.numPartitions && this.child == other.child
      case _ =>
        false
    }
  }

  override def hashCode(): Int = Objects.hashCode(numPartitions: java.lang.Integer, child)
}
