/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iotdb.confignode.procedure.impl.sync;

import org.apache.iotdb.common.rpc.thrift.TDataNodeConfiguration;
import org.apache.iotdb.common.rpc.thrift.TDataNodeLocation;
import org.apache.iotdb.common.rpc.thrift.TEndPoint;
import org.apache.iotdb.common.rpc.thrift.TNodeResource;
import org.apache.iotdb.commons.auth.entity.PrivilegeType;
import org.apache.iotdb.commons.path.PartialPath;
import org.apache.iotdb.confignode.consensus.request.ConfigPhysicalPlanType;
import org.apache.iotdb.confignode.consensus.request.write.auth.AuthorRelationalPlan;
import org.apache.iotdb.confignode.consensus.request.write.auth.AuthorTreePlan;
import org.apache.iotdb.confignode.procedure.store.ProcedureFactory;

import org.apache.tsfile.utils.PublicBAOS;
import org.junit.Assert;
import org.junit.Test;

import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;

import static org.junit.Assert.fail;

public class AuthOperationProcedureTest {

  @Test
  public void serializeDeserializeTest() throws IOException {
    final PublicBAOS byteArrayOutputStream = new PublicBAOS();
    final DataOutputStream outputStream = new DataOutputStream(byteArrayOutputStream);

    final TDataNodeLocation dataNodeLocation = new TDataNodeLocation();
    dataNodeLocation.setDataNodeId(1);
    dataNodeLocation.setClientRpcEndPoint(new TEndPoint("0.0.0.0", 6667));
    dataNodeLocation.setInternalEndPoint(new TEndPoint("0.0.0.0", 10730));
    dataNodeLocation.setMPPDataExchangeEndPoint(new TEndPoint("0.0.0.0", 10740));
    dataNodeLocation.setDataRegionConsensusEndPoint(new TEndPoint("0.0.0.0", 10760));
    dataNodeLocation.setSchemaRegionConsensusEndPoint(new TEndPoint("0.0.0.0", 10750));

    final TDataNodeConfiguration dataNodeConfiguration = new TDataNodeConfiguration();
    dataNodeConfiguration.setLocation(dataNodeLocation);
    dataNodeConfiguration.setResource(new TNodeResource(16, 34359738368L));

    final List<TDataNodeConfiguration> datanodes = new ArrayList<>();
    datanodes.add(dataNodeConfiguration);

    try {
      final int begin = ConfigPhysicalPlanType.CreateUser.ordinal();
      final int end = ConfigPhysicalPlanType.UpdateUser.ordinal();
      for (int i = begin; i <= end; i++) {
        final AuthOperationProcedure proc =
            new AuthOperationProcedure(
                new AuthorTreePlan(
                    ConfigPhysicalPlanType.values()[i],
                    "user1",
                    "role1",
                    "123456",
                    "123456",
                    Collections.singleton(1),
                    false,
                    Collections.singletonList(new PartialPath("root.t1"))),
                datanodes,
                false);
        proc.serialize(outputStream);
        final ByteBuffer buffer =
            ByteBuffer.wrap(byteArrayOutputStream.getBuf(), 0, byteArrayOutputStream.size());

        final AuthOperationProcedure proc2 =
            (AuthOperationProcedure) ProcedureFactory.getInstance().create(buffer);
        Assert.assertEquals(proc, proc2);
        buffer.clear();
        byteArrayOutputStream.reset();
      }
    } catch (final Exception e) {
      e.printStackTrace();
      fail();
    }

    try {
      final int begin = ConfigPhysicalPlanType.RCreateUser.ordinal();
      final int end = ConfigPhysicalPlanType.RRevokeRoleSysPri.ordinal();
      for (int i = begin; i <= end; i++) {
        final AuthOperationProcedure proc =
            new AuthOperationProcedure(
                new AuthorRelationalPlan(
                    ConfigPhysicalPlanType.values()[i],
                    "user1",
                    "role1",
                    "database",
                    "table",
                    new HashSet<>(PrivilegeType.CREATE.ordinal(), PrivilegeType.SELECT.ordinal()),
                    false,
                    "password123456"),
                datanodes,
                false);
        proc.serialize(outputStream);
        final ByteBuffer buffer =
            ByteBuffer.wrap(byteArrayOutputStream.getBuf(), 0, byteArrayOutputStream.size());

        final AuthOperationProcedure proc2 =
            (AuthOperationProcedure) ProcedureFactory.getInstance().create(buffer);
        Assert.assertEquals(proc, proc2);
        buffer.clear();
        byteArrayOutputStream.reset();
      }
    } catch (final Exception e) {
      e.printStackTrace();
      fail();
    }
  }
}
